/**
 * DB Converter Pro - Frontend Logic
 * Handles AJAX requests to the PHP backend
 */

const API_URL = 'api.php'; // The PHP Router file (Part 2)

// 1. Handle Access File Upload [cite: 8, 37]
document.getElementById('accessFile').addEventListener('change', async function (e) {
    if (e.target.files.length === 0) return;

    const formData = new FormData();
    formData.append('access_db', e.target.files[0]);
    formData.append('action', 'upload_access');

    showAlert('info', 'Uploading and reading database...');

    try {
        const response = await fetch(API_URL, { method: 'POST', body: formData });
        const result = await response.json();

        if (result.success) {
            showAlert('success', 'Database uploaded successfully.');
            loadTables(); // Refresh table list immediately
        } else {
            showAlert('danger', result.message);
        }
    } catch (error) {
        console.error(error);
        showAlert('danger', 'Upload failed. Check console for details.');
    }
});

// 2. Handle MySQL Connection [cite: 15, 38]
async function saveSqlConfig() {
    const form = document.getElementById('sqlForm');
    const formData = new FormData(form);
    formData.append('action', 'connect_mysql');

    try {
        const response = await fetch(API_URL, { method: 'POST', body: formData });
        const result = await response.json();

        if (result.success) {
            const modalEl = document.getElementById('sqlModal');
            const modal = bootstrap.Modal.getInstance(modalEl);
            modal.hide();
            showAlert('success', 'Connected to MySQL Server successfully.');
        } else {
            alert('Connection Failed: ' + result.message);
        }
    } catch (error) {
        alert('Network error connecting to MySQL.');
    }
}

// 3. Load Tables List [cite: 10]
async function loadTables() {
    try {
        const response = await fetch(`${API_URL}?action=get_tables`);
        const result = await response.json();

        const listContainer = document.getElementById('tableList');
        listContainer.innerHTML = ''; // Clear current list

        if (result.success && result.tables.length > 0) {
            result.tables.forEach(table => {
                const label = document.createElement('label');
                label.className = 'list-group-item table-list-item';
                label.innerHTML = `
                    <input class="form-check-input me-1 table-checkbox" type="checkbox" value="${table}">
                    ${table}
                `;
                // Add click event to preview data when clicking the row
                label.addEventListener('click', (e) => {
                    if (e.target.type !== 'checkbox') loadPreview(table);
                });
                listContainer.appendChild(label);
            });
        } else {
            listContainer.innerHTML = '<div class="p-2 text-muted">No tables found.</div>';
        }
    } catch (error) {
        console.error('Error loading tables:', error);
    }
}

// 4. Load Data Preview [cite: 39]
async function loadPreview(tableName) {
    document.getElementById('previewBody').innerHTML = '<tr><td colspan="4" class="text-center">Loading...</td></tr>';

    try {
        const response = await fetch(`${API_URL}?action=preview&table=${tableName}`);
        const result = await response.json();

        if (result.success) {
            renderPreviewTable(result.data);
        } else {
            document.getElementById('previewBody').innerHTML = `<tr><td colspan="4" class="text-danger">${result.message}</td></tr>`;
        }
    } catch (error) {
        console.error('Preview error:', error);
    }
}

// Helper to render the HTML table dynamically
function renderPreviewTable(data) {
    if (data.length === 0) return;

    // Generate Headers
    const headers = Object.keys(data[0]);
    let theadHtml = '<tr>';
    headers.forEach(h => theadHtml += `<th>${h}</th>`);
    theadHtml += '</tr>';
    document.querySelector('.preview-panel thead').innerHTML = theadHtml;

    // Generate Body
    let tbodyHtml = '';
    data.forEach(row => {
        tbodyHtml += '<tr>';
        Object.values(row).forEach(val => tbodyHtml += `<td>${val}</td>`);
        tbodyHtml += '</tr>';
    });
    document.getElementById('previewBody').innerHTML = tbodyHtml;
}

// 5. Handle Conversion Process [cite: 41, 12, 13]
async function convertData(mode) {
    // Collect selected tables if mode is 'selected'
    let selectedTables = [];
    if (mode === 'selected') {
        document.querySelectorAll('.table-checkbox:checked').forEach(cb => {
            selectedTables.push(cb.value);
        });
        if (selectedTables.length === 0) {
            showAlert('warning', 'Please select at least one table.');
            return;
        }
    }

    // UI Updates
    document.getElementById('progress-container').style.display = 'block';
    const progressBar = document.querySelector('.progress-bar');
    progressBar.style.width = '10%'; // Start
    document.getElementById('progress-text').innerText = 'Starting conversion...';

    const payload = new FormData();
    payload.append('action', 'convert');
    payload.append('mode', mode); // 'full', 'selected', 'to_mysql', etc.
    payload.append('tables', JSON.stringify(selectedTables));

    // Hide previous log button
    document.getElementById('btnDownloadLog').classList.add('d-none');

    try {
        const response = await fetch(API_URL, { method: 'POST', body: payload });
        const result = await response.json();

        progressBar.style.width = '100%';

        if (result.success) {
            progressBar.classList.add('bg-success');
            document.getElementById('progress-text').innerText = 'Done!';
            showAlert('success', result.message);
        } else {
            // IF ERRORS EXIST:
            progressBar.classList.add('bg-warning'); // Yellow for warning
            document.getElementById('progress-text').innerText = 'Completed with errors.';

            // Show the download button
            document.getElementById('btnDownloadLog').classList.remove('d-none');

            showAlert('warning', 'Conversion completed with errors. Download the log for details.');
        }
    } catch (error) {
        console.error(error);
        showAlert('danger', 'System error during conversion.');
    }
}

// 6. Utility: Notifications
function showAlert(type, message) {
    const alertArea = document.getElementById('alert-area');
    const wrapper = document.createElement('div');
    wrapper.innerHTML = [
        `<div class="alert alert-${type} alert-dismissible" role="alert">`,
        `   <div>${message}</div>`,
        '   <button type="button" class="btn-close" data-bs-dismiss="alert"></button>',
        '</div>'
    ].join('');
    alertArea.append(wrapper);

    // Auto dismiss after 3 seconds
    setTimeout(() => {
        wrapper.remove();
    }, 3000);
}

// 7. Bridge for HTML buttons (ADD THIS TO END OF script.js)
function refreshTables() {
    loadTables();
    showAlert('info', 'Refreshing table list...');
}

async function validateSelectedData() {
    // Get selected tables
    let selectedTables = [];
    document.querySelectorAll('.table-checkbox:checked').forEach(cb => {
        selectedTables.push(cb.value);
    });

    if (selectedTables.length === 0) {
        showAlert('warning', 'Please select tables to validate.');
        return;
    }

    showAlert('info', 'Validating data integrity...');

    const payload = new FormData();
    payload.append('action', 'validate');
    payload.append('tables', JSON.stringify(selectedTables));

    try {
        const response = await fetch(API_URL, { method: 'POST', body: payload });
        const result = await response.json();

        if (result.success) {
            showAlert('success', result.message);
        } else {
            // Show Detailed Report
            let reportHtml = '';
            for (const [table, issues] of Object.entries(result.report)) {
                reportHtml += `<h6>${table}</h6><ul>`;
                issues.forEach(issue => reportHtml += `<li class="text-danger">${issue}</li>`);
                reportHtml += `</ul>`;
            }

            // Inject into the Alert Area or a specific Modal
            const alertArea = document.getElementById('alert-area');
            alertArea.innerHTML = `
                <div class="alert alert-warning alert-dismissible" role="alert">
                    <h4 class="alert-heading">Validation Issues Found</h4>
                    <hr>
                    <div style="max-height: 200px; overflow-y: auto;">
                        ${reportHtml}
                    </div>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `;
        }
    } catch (error) {
        console.error(error);
        showAlert('danger', 'Validation failed.');
    }
}