<?php
// functions.php - COMPLETE VERSION

// 1. Connect to Access Database
function connectToAccess($filePath) {
    if (!file_exists($filePath)) {
        throw new Exception("Access database file not found.");
    }
    // PDO Connection String for Access (requires ODBC driver)
    $dsn = "odbc:DRIVER={Microsoft Access Driver (*.mdb, *.accdb)};Dbq=$filePath;Uid=;Pwd=;";
    try {
        $pdo = new PDO($dsn);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    } catch (PDOException $e) {
        error_log("Access Connection Error: " . $e->getMessage());
        throw new Exception("Connection failed. Ensure Microsoft Access Database Engine is installed.");
    }
}

// 2. Read Tables from Access
function getAccessTables($pdo) {
    // Query MSysObjects to find user tables (Type 1)
    $sql = "SELECT Name FROM MSysObjects WHERE Type=1 AND Flags=0";
    try {
        $stmt = $pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (Exception $e) {
        // Show the specific error in the list so we know what is wrong
        return ["ERROR: " . $e->getMessage()];
    }
}

// 3. Reverse Conversion (MySQL -> Access)
function exportToAccess($mysqlPdo, $tableName) {
    $template = __DIR__ . '/templates/blank.mdb';
    $newDbName = 'export_' . time() . '_' . $tableName . '.mdb';
    $newDbPath = __DIR__ . '/uploads/' . $newDbName;
    
    // Copy blank template
    if (!copy($template, $newDbPath)) {
        throw new Exception("Failed to create new Access file from template.");
    }

    $accessPdo = connectToAccess($newDbPath);

    // Get Data from MySQL
    $stmt = $mysqlPdo->query("SELECT * FROM `$tableName`");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($rows)) return $newDbPath;

    // Create Table in Access
    $columns = array_keys($rows[0]);
    $colDefs = implode(" TEXT, ", $columns) . " TEXT"; 
    $accessPdo->exec("CREATE TABLE $tableName ($colDefs)");

    // Insert Data
    $placeholders = implode(", ", array_fill(0, count($columns), "?"));
    $insertSql = "INSERT INTO $tableName VALUES ($placeholders)";
    $insertStmt = $accessPdo->prepare($insertSql);

    foreach ($rows as $row) {
        $insertStmt->execute(array_values($row));
    }
    
    return $newDbPath;
}

// 4. Validation Logic
function validateTable($accessPdo, $tableName) {
    $issues = [];
    $stmt = $accessPdo->query("SELECT TOP 1 * FROM [$tableName]");
    $columns = array_keys($stmt->fetch(PDO::FETCH_ASSOC) ?: []);
    
    foreach ($columns as $col) {
        if (preg_match('/[^a-zA-Z0-9_]/', $col)) {
            $issues[] = "Column '$col': Contains spaces or special characters.";
        }
    }
    return $issues;
}

// 5. Logging Helpers
function getLogFilePath() {
    if (session_status() === PHP_SESSION_NONE) session_start();
    $sessionId = session_id();
    return __DIR__ . "/uploads/log_{$sessionId}.txt";
}

function initErrorLog() {
    $file = getLogFilePath();
    $header = "Conversion Log - Started at " . date('Y-m-d H:i:s') . "\n------------------\n";
    file_put_contents($file, $header);
}

function logError($context, $message) {
    $file = getLogFilePath();
    $entry = "[" . date('H:i:s') . "] [$context] Error: $message\n";
    file_put_contents($file, $entry, FILE_APPEND);
}

// 6. Conversion Logic with Logging
function convertTableToMySQL_WithLogging($accessPdo, $mysqlPdo, $tableName) {
    try {
        $stmt = $accessPdo->query("SELECT * FROM [$tableName]");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (empty($rows)) {
            logError($tableName, "Skipped - Table is empty.");
            return;
        }

        // Create Table (Simplified)
        $columns = array_keys($rows[0]);
        $mysqlPdo->exec("CREATE TABLE IF NOT EXISTS `$tableName` (id INT AUTO_INCREMENT PRIMARY KEY, " . implode(" TEXT, ", $columns) . " TEXT)");

        // Insert
        $placeholders = implode(", ", array_fill(0, count($columns), "?"));
        $insertStmt = $mysqlPdo->prepare("INSERT INTO `$tableName` (`" . implode("`, `", $columns) . "`) VALUES ($placeholders)");

        foreach ($rows as $index => $row) {
            try {
                $insertStmt->execute(array_values($row));
            } catch (Exception $e) {
                logError($tableName, "Row $index Failed: " . $e->getMessage());
            }
        }
    } catch (Exception $e) {
        logError($tableName, "Critical Table Failure: " . $e->getMessage());
    }
}
?>