<?php
/**
 * API Router for DB Converter Pro
 * Handles requests from script.js and returns JSON.
 */

// Enable Error Reporting for Debugging (Disable in Production)
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't print errors to output, breaks JSON
header('Content-Type: application/json');

session_start();
require_once 'functions.php'; // Assume functions from Step 4 are here

$response = ['success' => false, 'message' => 'Invalid Request'];

try {
    // determine action from GET or POST
    $action = $_REQUEST['action'] ?? '';

    switch ($action) {
        
        case 'upload_access': // [cite: 8]
            if (!isset($_FILES['access_db'])) throw new Exception("No file uploaded.");
            
            $targetDir = __DIR__ . '/uploads/';
            if (!is_dir($targetDir)) mkdir($targetDir, 0777, true);
            
            $fileName = "current_db.accdb"; // Save as a standard name for session
            $targetPath = $targetDir . $fileName;
            
            if (move_uploaded_file($_FILES['access_db']['tmp_name'], $targetPath)) {
                $_SESSION['access_path'] = $targetPath; // Store path in session
                $response = ['success' => true, 'message' => 'File uploaded.'];
            } else {
                throw new Exception("File move failed.");
            }
            break;

        case 'connect_mysql': // [cite: 15]
            // Store credentials in session (In production, encrypt this or use env)
            $_SESSION['mysql_config'] = [
                'host' => $_POST['host'],
                'user' => $_POST['user'],
                'pass' => $_POST['pass'],
                'dbname' => $_POST['dbname']
            ];
            
            // Test Connection
            $pdo = getMysqlConnection(); // defined in functions.php
            $response = ['success' => true, 'message' => 'Connected successfully.'];
            break;

        case 'get_tables': // [cite: 9]
            if (!isset($_SESSION['access_path'])) throw new Exception("No Access DB uploaded.");
            
            $pdo = connectToAccess($_SESSION['access_path']);
            $tables = getAccessTables($pdo); // defined in functions.php
            
            $response = ['success' => true, 'tables' => $tables];
            break;

        case 'preview': // [cite: 10]
            $table = $_GET['table'] ?? '';
            if (!$table) throw new Exception("No table specified.");
            
            $pdo = connectToAccess($_SESSION['access_path']);
            // Limit to 10 rows for preview
            $stmt = $pdo->prepare("SELECT TOP 10 * FROM [$table]");
            $stmt->execute();
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = ['success' => true, 'data' => $data];
            break;

        case 'convert': 
            $mode = $_POST['mode'];
            $tablesToConvert = [];
            $results = [];

            // 1. Identify Source and Destination
            if ($mode === 'to_access') {
                // Direction: MySQL -> Access
                $sourcePdo = getMysqlConnection();
                // For reverse, we might need a list of MySQL tables. 
                // Simplified: We assume user selected tables from the UI list (which currently shows Access tables).
                // In a full reverse mode, you'd populate the UI with MySQL tables first. 
                // For this example, we will convert the tables selected in the UI.
                $tablesToConvert = json_decode($_POST['tables'], true); 
                
                foreach ($tablesToConvert as $table) {
                    // Call the function defined in Step 4 (exportToAccess)
                    $newFile = exportToAccess($sourcePdo, $table);
                    $results[] = "Exported $table to $newFile";
                }
                 $response = ['success' => true, 'message' => "Export successful.", 'details' => $results];

            } else {
                // Direction: Access -> MySQL (Default)
                $accessPdo = connectToAccess($_SESSION['access_path']);
                $mysqlPdo = getMysqlConnection();
                
                // Initialize Logging
                initErrorLog();

                if ($mode === 'selected') {
                    $tablesToConvert = json_decode($_POST['tables'], true);
                } else {
                    $tablesToConvert = getAccessTables($accessPdo);
                }

                $count = 0;
                foreach ($tablesToConvert as $table) {
                    convertTableToMySQL_WithLogging($accessPdo, $mysqlPdo, $table);
                    $count++;
                }
                $response = ['success' => true, 'message' => "Successfully converted $count tables to MySQL."];
            }
            break;

        case 'download_log':
            $file = getLogFilePath();
            
            if (file_exists($file)) {
                header('Content-Description: File Transfer');
                header('Content-Type: application/octet-stream');
                header('Content-Disposition: attachment; filename="conversion_log.txt"');
                header('Expires: 0');
                header('Cache-Control: must-revalidate');
                header('Pragma: public');
                header('Content-Length: ' . filesize($file));
                readfile($file);
                exit;
            } else {
                http_response_code(404);
                echo "Log file not found.";
            }
            break;

        case 'validate':
            $pdo = connectToAccess($_SESSION['access_path']);
            $tables = json_decode($_POST['tables'], true);
            $report = [];

            foreach ($tables as $table) {
                $tableIssues = validateTable($pdo, $table);
                if (!empty($tableIssues)) {
                    $report[$table] = $tableIssues;
                }
            }

            if (empty($report)) {
                $response = ['success' => true, 'message' => 'Validation Passed: Data looks good!'];
            } else {
                $response = ['success' => false, 'report' => $report];
            }
            break;

        default:
            throw new Exception("Unknown action: " . $action);
    }

} catch (Exception $e) {
    // Global Error Handling 
    $response = [
        'success' => false,
        'message' => $e->getMessage()
    ];
}

echo json_encode($response);
exit;

// --- Helper for MySQL Connection inside API ---
function getMysqlConnection() {
    if (!isset($_SESSION['mysql_config'])) throw new Exception("MySQL not configured.");
    $conf = $_SESSION['mysql_config'];
    
    $dsn = "mysql:host={$conf['host']};dbname={$conf['dbname']};charset=utf8mb4";
    return new PDO($dsn, $conf['user'], $conf['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
    ]);
}
?>